/**
 * \file: UspiInputSourceImpl.cpp
 *
 * \version: $Id:$
 *
 * \release: $Name:$
 *
 * <brief description>.
 * <detailed description>
 * \component: Android Auto
 *
 * \author: M. Adachi / ADIT/SW / madachi@jp.adit-jv.com
 *          P. Acar / ADIT/SW2 / pacar@de.adit-jv.com
 *
 * \copyright (c) 2016 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 * \see <related items>
 *
 * \history
 *
 ***********************************************************************/
#include <string>
#include <adit_logging.h>
#include <uspi/WaylandFacade.h>
#include <uspi/ITouchFacade.h>
#include "UspiInputSourceImpl.h"

LOG_IMPORT_CONTEXT(aauto_input)

namespace adit { namespace aauto {

using namespace uspi;
using std::string;
using std::pair;
using std::unique_ptr;

UspiInputSourceImpl::UspiInputSourceImpl(InputSource* inInputSource, void* inSessionContext)
: mInputSource(inInputSource), mSessionContext(inSessionContext), mCallbacks(nullptr), mTouchFacade(nullptr),
  mMarginX(0), mMarginY(0), mRunning(false) { }

UspiInputSourceImpl::~UspiInputSourceImpl()
{
    if (true == mRunning) {
        if (!shutdown()) {
            LOG_WARN((aauto_input, "shutdown() failed"));
        }
    }
    LOGD_DEBUG((aauto_input, "UspiInputSource de-initialize was done."));
}

bool UspiInputSourceImpl::init()
{
    // Check configuration error
    if(!mConfig.ResultConfig())
    {
        LOG_ERROR((aauto_input, "Check configuration error"));
        return false;
    }

    mTouchFacade = move(unique_ptr<WaylandFacade>((new WaylandFacade(mSessionContext))->
            setDisplaySize(mConfig.displayWidth, mConfig.displayHeight)->
            setFingerNum(2)));

    if(!mTouchFacade->initialize(this))
    {
        LOG_ERROR((aauto_input, "Failed to initialize touch facade"));
        return false;
    }

    mRunning = true;
    mInputSource->registerCallbacks(this);

    return true;
}

bool UspiInputSourceImpl::shutdown()
{
    bool ret = true;

    if (true == mRunning) {
        if (!mTouchFacade->deinitialize())
        {
            LOG_ERROR((aauto_input, "Failed to deinitialize touch facade"));
            ret = false;
        } else {
            mRunning = false;
        }
    } else {
        LOG_INFO((aauto_input, "UspiInputSourceImpl::shutdown()  Not running. Was already stopped"));
    }

    LOGD_DEBUG((aauto_input, "UspiInputSourceImpl shutdown done"));

    return ret;
}

void UspiInputSourceImpl::setConfigItem(string inKey, string inValue)
{
    LOGD_DEBUG((aauto_input, "setConfigItem: inKey = %s inValue = %s", inKey.c_str(),inValue.c_str())); 
    mConfig.set(inKey, inValue);
}

void UspiInputSourceImpl::registerCallbacks(IAditInputSourceCallbacks* inCallbacks)
{
    mCallbacks = inCallbacks;
}

void UspiInputSourceImpl::onTouch(TouchEvent inEvent)
{
    LOGD_VERBOSE((aauto_input, "entry: type: %d, finger: %d, x: %f, y: %f",
            inEvent.eventType, inEvent.fingerId, inEvent.xRelative, inEvent.yRelative));

    int x_pixel = (inEvent.xRelative * mConfig.displayWidth) - mMarginX;
    int y_pixel = (inEvent.yRelative * mConfig.displayHeight) - mMarginY;

    auto it = mDownFingers.find(inEvent.fingerId);
    // finger touching display, therefore tracked
    if (it != mDownFingers.end())
    {
        switch (inEvent.eventType)
        {
        case TouchEvent::down:
            LOG_WARN((aauto_input, "Touch down: Didn't catch any up event"));
            it->second.xPixel = x_pixel;
            it->second.yPixel = y_pixel;
            if (mDownFingers.size() == 1) {
                notifyTouch(ACTION_DOWN, inEvent.fingerId);
            } else {
                notifyTouch(ACTION_POINTER_DOWN, inEvent.fingerId);
            }
            break;
        case TouchEvent::move:
            LOGD_VERBOSE((aauto_input, "Touch move event"));
            it->second.xPixel = x_pixel;
            it->second.yPixel = y_pixel;
            notifyTouch(ACTION_MOVED, inEvent.fingerId);
            break;
        case TouchEvent::up:
            LOGD_VERBOSE((aauto_input, "Touch up event"));
            if (mDownFingers.size() == 1) {
                notifyTouch(ACTION_UP, inEvent.fingerId);
            } else {
                notifyTouch(ACTION_POINTER_UP, inEvent.fingerId);
            }
            mDownFingers.erase(inEvent.fingerId);
            break;
        }
    // new finger
    } else {
        switch (inEvent.eventType)
        {
        case TouchEvent::down:
            LOGD_VERBOSE((aauto_input, "Touch down: New finger"));
            mDownFingers.insert(pair<int, Coordinates>(inEvent.fingerId, {x_pixel, y_pixel}));
            if (mDownFingers.size() == 1) {
                notifyTouch(ACTION_DOWN, inEvent.fingerId);
            } else {
                notifyTouch(ACTION_POINTER_DOWN, inEvent.fingerId);
            }
            break;
        case TouchEvent::move:
            LOG_WARN((aauto_input, "Touch move: Finger can't move when it wasn't contacting the screen"));
            break;
        case TouchEvent::up:
            LOG_WARN((aauto_input, "Touch up: Finger can't be lifted when it wasn't contacting the screen"));
            break;
        }
    }

    if (mConfig.verbose)
    {
        LOGD_VERBOSE((aauto_input, "Can touch this at %f, %f finger: %d, pressure: %f",
                inEvent.xRelative, inEvent.yRelative, inEvent.fingerId, inEvent.pressure));
    }
}

bool UspiInputSourceImpl::notifyTouch(int inActionType, int inFingerIndex)
{
    struct timespec timestamp;
    clock_gettime(CLOCK_MONOTONIC, &timestamp);

    uint32_t x[mDownFingers.size()];
    uint32_t y[mDownFingers.size()];
    uint32_t id[mDownFingers.size()];
    int action_index = 0;

    int event_index = 0;
    for (auto it = mDownFingers.begin(); it != mDownFingers.end(); it++, event_index++)
    {
        x[event_index] = it->second.xPixel;
        y[event_index] = it->second.yPixel;
        id[event_index] = it->first;
        if(inFingerIndex == it->first)
        {
            action_index = event_index;
        }
    }

    mInputSource->reportTouch(timestamp.tv_sec * 1000000, mDownFingers.size(),
            id, x, y, inActionType, action_index);

    return true;
}

void UspiInputSourceImpl::onTouchError(int inError)
{
    if (mCallbacks != nullptr)
    {
        // registering for input callbacks is optional, at least for now
        mCallbacks->notifyErrorCallback((aautoErrorCodes)inError);
    } else {
        LOGD_DEBUG((aauto_input, "No callbacks registered"));
    }
}

void UspiInputSourceImpl::onLogging(UspiLogLevel inLogLevel, const string& inLogString)
{
    switch (inLogLevel) {
        case USPI_LOG_FATAL:
            LOG_FATAL((aauto_input, "%s", inLogString.c_str()));
            break;
        case USPI_LOG_ERROR:
            LOG_ERROR((aauto_input, "%s", inLogString.c_str()));
            break;
        case USPI_LOG_WARN:
            LOG_WARN((aauto_input, "%s", inLogString.c_str()));
            break;
        case USPI_LOG_INFO:
            LOG_INFO((aauto_input, "%s", inLogString.c_str()));
            break;
        case USPI_LOG_DEBUG:
            LOGD_DEBUG((aauto_input, "%s", inLogString.c_str()));
            break;
        case USPI_LOG_VERBOSE:
            if (mConfig.verbose) {
                LOGD_VERBOSE((aauto_input, "%s", inLogString.c_str()));
            }
            break;
        default:
            break;
    }
}

void UspiInputSourceImpl::onLateResolutionData(int inWidth, int inHeight, int inMarginX, int inMarginY)
{
    mConfig.displayWidth = inWidth;
    mConfig.displayHeight = inHeight;
    mMarginX = inMarginX;
    mMarginY = inMarginY;

    LOGD_DEBUG((aauto_input, "%s()  Set margin to %d:%d for resolution %dx%d",
                __func__, mMarginX, mMarginY, mConfig.displayWidth, mConfig.displayHeight));
}
void UspiInputSourceImpl::onInputFeedback(const InputFeedback& inFeedback)
{
    if (mCallbacks != nullptr)
    {
        mCallbacks->onInputFeedback(inFeedback);
    } else {
        // registering for input callbacks is optional, at least for now
        LOGD_DEBUG((aauto_input, "No callbacks registered"));
    }
}

} } // namespace adit { namespace aauto {
